/* scp_tty.c: Operating system-dependent terminal I/O routines.

   Copyright 1994, Robert M Supnik, Digital Equipment Corporation
   Commercial use prohibited

   ttinit	-	called once to get initial terminal state
   ttrunstate	-	called to put terminal into run state
   ttcmdstate	-	called to return terminal to command state
   ttclose	-	called once before the simulator exits
   sim_poll_kbd	-	poll for keyboard input, -1 if no input
   sim_type_tty -	output character to terminal

*/

/*
 SCP_TTY.C
 Revision 1.5  1995/02/02 03:29:53  T.V.Kutergin
 Fixed bug in KOI7->ASCII conversion table
 Added RAW/KOI7 switch for paper tape puncher/reader (for text files)

 Revision 1.4  1995/01/29 02:11:06  T.V.Kutergin
 Added KOI7 support and CR/LF expansion for LPT.

 Revision 1.3  1995/01/28 23:08:34  T.V.Kutergin
 Added ASCII / KOI7 mode switching for TTI and TTO (for Russian users)
 Commands: SET TTI ASCII/KOI7  and SET TTO ASCII/KOI7

 Revision 1.2  1995/01/23 16:56:39  T.V.Kutergin
 Implemented partial VT52 emulation

 Revision 1.1  1995/01/22 05:10:48  T.V.Kutergin
 Initial revision

*/

#include "sim_defs.h"
#include <pc.h>
#include <gppconio.h>
#include <keys.h>
#include <dos.h>

#ifdef  RCS_INVOKED
static  char* rcsid="SCP_TTY.C 1.5 1995/02/02 03:29:53 T.V.Kutergin Exp";
#endif

/*----------------------------------------------------------------*/
#define MODE_COLOR      C80
#define MODE_BW         BW80
#define ATTR_COLOR       7
#define ATTR_MONO        7
#define MIN_ROW          1
#define MIN_COL          1
#define MAX_ROW         25
#define MAX_COL         80
#define TAB_SIZE         8
#define ESCSEQ_MAX      10

#define ASCII_CTL_MAX   0x1F
#define ASCII_MAX       0xFF

#ifdef  KOI7
extern  UNIT            tti_unit,tto_unit;
#define IsTtiKoi7()     (tti_unit.flags & UNIT_KOI7)
#define IsTtoKoi7()     (tto_unit.flags & UNIT_KOI7)
#endif

typedef struct                     /* special key sequence        */
{
   int            key;
   unsigned char* seq;
} INKEYSEQ;

typedef struct                     /* control key implementation  */
{
   int            key;
   void           (*ctlfunc)(void);
   Boolean        canesc;
} CTLFUNC;


typedef struct                     /* escape seq. implementation  */
{
   unsigned char* seq;
   void           (*ctlfunc)(void);
   unsigned short getchars;
} ESCFUNC;

static void call_null(void);
static void ctl_bell(void);
static void ctl_bs(void);
static void ctl_ht(void);
static void ctl_lf(void);
static void ctl_vt(void);
static void ctl_ff(void);
static void ctl_cr(void);
static void ctl_can(void);
static void ctl_sub(void);
static void ctl_esc(void);

static void esc_a(void);
static void esc_b(void);
static void esc_c(void);
static void esc_d(void);
static void esc_h(void);
static void esc_i(void);
static void esc_j(void);
static void esc_k(void);
static void esc_y(void);
static void esc_z(void);
static void esc_eq(void);
static void esc_gt(void);
static void esc_lbrk(void);
static void esc_slash(void);

/*----------------------------------------------------------------*/
static INKEYSEQ inkey[]={          /* translate key to sequence   */
                         {K_Up              ,"\033A"},
                         {K_Down            ,"\033B"},
                         {K_Right           ,"\033C"},
                         {K_Left            ,"\033D"},
                         {K_EUp             ,"\033A"},
                         {K_EDown           ,"\033B"},
                         {K_ERight          ,"\033C"},
                         {K_Eleft           ,"\033D"},
                         {K_F1              ,"\033P"},
                         {K_F2              ,"\033Q"},
                         {K_F3              ,"\033R"},
                         {K_F4              ,"\033S"},
                         {K_Control_Insert  ,"\033?p"},
                         {K_Control_End     ,"\033?q"},
                         {K_Control_Down    ,"\033?r"},
                         {K_Control_PageDown,"\033?s"},
                         {K_Control_Left    ,"\033?t"},
                         {K_Control_Center  ,"\033?u"},
                         {K_Control_Right   ,"\033?v"},
                         {K_Control_Home    ,"\033?w"},
                         {K_Control_Up      ,"\033?x"},
                         {K_Control_PageUp  ,"\033?y"},
                         {K_Control_Delete  ,"\033?n"},
                         {K_Control_KPDash  ,"\033?m"},
                         {K_Control_KPPlus  ,"\033?l"},
                         {0x153             ,"\x07F"},
                         {0x253             ,"\x07F"},
                        };

static CTLFUNC ctlfnc[]={          /* control key impl.table      */
                         {0x07,ctl_bell,FALSE},
                         {0x08,ctl_bs  ,FALSE},
                         {0x09,ctl_ht  ,FALSE},
                         {0x0A,ctl_lf  ,FALSE},
                         {0x0B,ctl_vt  ,FALSE},
                         {0x0C,ctl_ff  ,FALSE},
                         {0x0D,ctl_cr  ,FALSE},
                         {0x18,ctl_can ,TRUE},
                         {0x1A,ctl_sub ,TRUE},
                         {0x1B,ctl_esc ,TRUE},
                        };

static ESCFUNC escfnc[]={          /* escape seq. impl.table      */
                         {"A",esc_a,0},
                         {"B",esc_b,0},
                         {"C",esc_c,0},
                         {"D",esc_d,0},
                         {"F",call_null,0},
                         {"G",call_null,0},
                         {"H",esc_h,0},
                         {"I",esc_i,0},
                         {"J",esc_j,0},
                         {"K",esc_k,0},
                         {"Y",esc_y,2},
                         {"Z",esc_z,0},
                         {"=",esc_eq,0},
                         {">",esc_gt,0},
                         {"<",call_null,0},
                         {"[",esc_lbrk,0},
                         {"/",esc_slash,0},
                         {"(",call_null,0},
                         {")",call_null,0},
                         {"W",call_null,0},
                         {"X",call_null,0},
                         {"^",call_null,0},
                         {"-",call_null,0},
                         {"]",call_null,0},
                         {"V",call_null,0},
                        };

#ifdef KOI7                        /* process russian letters   */
unsigned char pc2pdpK7[]={0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x7F,0x09,0x0A,0x0B,0x0C,0x0D,0x0E,0x0F,
                          0x10,0x11,0x12,0x13,0x14,0x15,0x16,0x17,0x18,0x19,0x1A,0x1B,0x1C,0x1D,0x1E,0x1F,
                          0x20,0x21,0x22,0x23,0x24,0x25,0x26,0x27,0x28,0x29,0x2A,0x2B,0x2C,0x2D,0x2E,0x2F,
                          0x30,0x31,0x32,0x33,0x34,0x35,0x36,0x37,0x38,0x39,0x3A,0x3B,0x3C,0x3D,0x3E,0x3F,
                          0x40,0x41,0x42,0x43,0x44,0x45,0x46,0x47,0x48,0x49,0x4A,0x4B,0x4C,0x4D,0x4E,0x4F,
                          0x50,0x51,0x52,0x53,0x54,0x55,0x56,0x57,0x58,0x59,0x5A,0x5B,0x5C,0x5D,0x5E,0x5F,
                          0x60,0x41,0x42,0x43,0x44,0x45,0x46,0x47,0x48,0x49,0x4A,0x4B,0x4C,0x4D,0x4E,0x4F,
                          0x50,0x51,0x52,0x53,0x54,0x55,0x56,0x57,0x58,0x59,0x5A,0x5B,0x5C,0x5D,0x5E,0x5F,
                          0x61,0x62,0x77,0x67,0x64,0x65,0x76,0x7A,0x69,0x6A,0x6B,0x6C,0x6D,0x6E,0x6F,0x70,
                          0x72,0x73,0x74,0x75,0x66,0x68,0x63,0x7E,0x7B,0x7D,0x78,0x79,0x78,0x7C,0x60,0x71,
                          0x61,0x62,0x77,0x67,0x64,0x65,0x76,0x7A,0x69,0x6A,0x6B,0x6C,0x6D,0x6E,0x6F,0x70,
                          0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                          0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                          0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                          0x72,0x73,0x74,0x75,0x66,0x68,0x63,0x7E,0x7B,0x7D,0x78,0x79,0x78,0x7C,0x60,0x71,
                          0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00};

unsigned char pdp2pcK7[]={0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0A,0x0B,0x0C,0x0D,0x0E,0x0F,
                          0x10,0x11,0x12,0x13,0x14,0x15,0x16,0x17,0x18,0x19,0x1A,0x1B,0x1C,0x1D,0x1E,0x1F,
                          0x20,0x21,0x22,0x23,0x24,0x25,0x26,0x27,0x28,0x29,0x2A,0x2B,0x2C,0x2D,0x2E,0x2F,
                          0x30,0x31,0x32,0x33,0x34,0x35,0x36,0x37,0x38,0x39,0x3A,0x3B,0x3C,0x3D,0x3E,0x3F,
                          0x40,0x41,0x42,0x43,0x44,0x45,0x46,0x47,0x48,0x49,0x4A,0x4B,0x4C,0x4D,0x4E,0x4F,
                          0x50,0x51,0x52,0x53,0x54,0x55,0x56,0x57,0x58,0x59,0x5A,0x5B,0x5C,0x5D,0x5E,0x5F,
                          0x9E,0x80,0x81,0x96,0x84,0x85,0x94,0x83,0x95,0x88,0x89,0x8A,0x8B,0x8C,0x8D,0x8E,
                          0x8F,0x9F,0x90,0x91,0x92,0x93,0x86,0x82,0x9C,0x9B,0x87,0x98,0x9D,0x99,0x97,0x7F,
                          0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                          0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                          0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                          0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                          0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                          0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                          0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                          0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00};
#endif
unsigned char pc2pdpAS[]={0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x7F,0x09,0x0A,0x0B,0x0C,0x0D,0x0E,0x0F,
                          0x10,0x11,0x12,0x13,0x14,0x15,0x16,0x17,0x18,0x19,0x1A,0x1B,0x1C,0x1D,0x1E,0x1F,
                          0x20,0x21,0x22,0x23,0x24,0x25,0x26,0x27,0x28,0x29,0x2A,0x2B,0x2C,0x2D,0x2E,0x2F,
                          0x30,0x31,0x32,0x33,0x34,0x35,0x36,0x37,0x38,0x39,0x3A,0x3B,0x3C,0x3D,0x3E,0x3F,
                          0x40,0x41,0x42,0x43,0x44,0x45,0x46,0x47,0x48,0x49,0x4A,0x4B,0x4C,0x4D,0x4E,0x4F,
                          0x50,0x51,0x52,0x53,0x54,0x55,0x56,0x57,0x58,0x59,0x5A,0x5B,0x5C,0x5D,0x5E,0x5F,
                          0x60,0x61,0x62,0x63,0x64,0x65,0x66,0x67,0x68,0x69,0x6A,0x6B,0x6C,0x6D,0x6E,0x6F,
                          0x70,0x71,0x72,0x73,0x74,0x75,0x76,0x77,0x78,0x79,0x7A,0x7B,0x7C,0x7D,0x7E,0x7F,
                          0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                          0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                          0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                          0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                          0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                          0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                          0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                          0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00};

unsigned char pdp2pcAS[]={0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0A,0x0B,0x0C,0x0D,0x0E,0x0F,
                          0x10,0x11,0x12,0x13,0x14,0x15,0x16,0x17,0x18,0x19,0x1A,0x1B,0x1C,0x1D,0x1E,0x1F,
                          0x20,0x21,0x22,0x23,0x24,0x25,0x26,0x27,0x28,0x29,0x2A,0x2B,0x2C,0x2D,0x2E,0x2F,
                          0x30,0x31,0x32,0x33,0x34,0x35,0x36,0x37,0x38,0x39,0x3A,0x3B,0x3C,0x3D,0x3E,0x3F,
                          0x40,0x41,0x42,0x43,0x44,0x45,0x46,0x47,0x48,0x49,0x4A,0x4B,0x4C,0x4D,0x4E,0x4F,
                          0x50,0x51,0x52,0x53,0x54,0x55,0x56,0x57,0x58,0x59,0x5A,0x5B,0x5C,0x5D,0x5E,0x5F,
                          0x60,0x61,0x62,0x63,0x64,0x65,0x66,0x67,0x68,0x69,0x6A,0x6B,0x6C,0x6D,0x6E,0x6F,
                          0x70,0x71,0x72,0x73,0x74,0x75,0x76,0x77,0x78,0x79,0x7A,0x7B,0x7C,0x7D,0x7E,0x7F,
                          0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                          0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                          0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                          0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                          0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                          0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                          0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
                          0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00};

/*----------------------------------------------------------------*/
int sim_int_char = KEY_STOP;          /* interrupt character      */
static unsigned char* inchar=NULL;    /* input sequence           */
static Boolean        runstate=FALSE; /* run state flag           */
static unsigned short attr;           /* screen attribute         */
/*----------------------------------------------------------------*/
static Boolean        escstate=FALSE; /* esc.sequence flag        */
static unsigned char  escseq[ESCSEQ_MAX];
static unsigned short epos,eget;
void                  (*armed)(void); /* run after all eget       */

/*----------------------------------------------------------------*/
unsigned char* key2sec(int key)
{short          i;
 unsigned char* seq;
   for (i=0,seq=NULL;
         (seq==NULL)&&(i<sizeof(inkey)/sizeof(inkey[0])); i++)
   {
      if (inkey[i].key==key) seq=inkey[i].seq;
   };
   return(seq);
}; /* key2sec */

/*----------------------------------------------------------------*/

int ttinit (void)
{
#ifdef MODE_COLOR
 unsigned   short mode;
 union REGS in,out;
#endif
   runstate=FALSE;
   gppconio_init();
#ifdef MODE_COLOR
   mode=(ScreenMode()==7) ? MODE_BW : MODE_COLOR;
   attr=(ScreenMode()==MODE_COLOR) ? ATTR_COLOR : ATTR_MONO;
   if ((mode!=ScreenMode()) ||
       (ScreenRows()!=(MAX_ROW-MIN_ROW+1)) ||
       (ScreenCols()!=(MAX_COL-MIN_COL+1)))
   {
      in.h.ah=0x0; in.h.al=mode;
      int86(0x10,&in,&out);
   };
   textattr(attr);
   clrscr(); gotoxy(MIN_COL,MIN_ROW);
#else
   attr=ATTR_MONO;
#ifdef CLS_ON_INIT
   textattr(ATTR_MONO);
   clrscr(); gotoxy(MIN_COL,MIN_ROW);
#endif
#endif
   return SCPE_OK;                 /* return success      */
}

int ttrunstate (void)
{
   runstate=TRUE;
   return SCPE_OK;
}

int ttcmdstate (void)
{
   runstate=FALSE;
   return SCPE_OK;
}

int ttclose (void)
{
#ifdef CLS_ON_INIT
   clrscr(); gotoxy(MIN_COL,MIN_ROW);
#endif
   return ttcmdstate();
}

int sim_poll_kbd (void)
{
 int   c,nc;
 short i;
   if (inchar!=NULL)
   {
      c=*inchar++;
      if (*inchar==0) inchar=NULL;
      return (c ? (c | SCPE_KFLAG) : SCPE_OK);
   }
   else if (kbhit())
   {
      c=getxkey();
      if (runstate && (c==sim_int_char))
         return(SCPE_STOP);
      else
      {
#ifdef KOI7

         if ((c==0x200) && IsTtiKoi7()) c=0xE0;
#endif
         if (c>ASCII_MAX)          /* Non ASCII       */
         {
            inchar=key2sec(c);
            if (inchar==NULL)
               c=0;
            else
            {
               c=*inchar++;
               if (*inchar==0) inchar=NULL;
            };
         }
         else                      /* ASCII character */
         {
#ifdef KOI7
            nc=IsTtiKoi7() ? pc2pdpK7[c] : pc2pdpAS[c];
#else
            nc=pc2pdpAS[c];
#endif
            if (nc==0)
            {
               inchar=key2sec(c);
               if (inchar==NULL)
                  c=0;
               else
               {
                  c=*inchar++;
                  if (*inchar==0) inchar=NULL;
               };
            }
            else
               c=nc;
         };
      };
      return (c ? (c | SCPE_KFLAG) : SCPE_OK);
   }
   else
      return(SCPE_OK);
}

int sim_type_tty (char out)
{short i;
 unsigned char c;
   if (escstate)                 /* escape seq in process */
   {
      if (out <= ASCII_CTL_MAX)  /* control character     */
      {
         for (i=0; i<(sizeof(ctlfnc)/sizeof(ctlfnc[0])); i++)
         {
            if (ctlfnc[i].key==out)
            {
               if (ctlfnc[i].canesc)
               {
                  (*ctlfnc[i].ctlfunc)();
                  out=0;
                  break;
               }
               else
                  break;
            };
         };
      };
      if (out!=0)                /* continue accept       */
      {
         if (eget > 0)           /* accept addition.chars */
         {
            escseq[epos++]=out; escseq[epos]=0;
            if (--eget==0)
            {
               (*armed)();
               escstate=FALSE; epos=eget=0; armed=NULL;
            };
         }
         else
         {
            if (epos < ESCSEQ_MAX)
            {
               escseq[epos++]=out; escseq[epos]=0;
               /* check escape seq. */
               for (i=0, armed=NULL;
                    (armed==NULL) && (i<(sizeof(escfnc)/sizeof(escfnc[0])));
                    i++)
               {
                  if (strcmp(escfnc[i].seq,escseq)==0)
                  {
                     armed=escfnc[i].ctlfunc; eget=escfnc[i].getchars;
                  };
               };
               if ((armed!=NULL) && (eget==0))
               {
                  (*armed)();
                  escstate=FALSE; epos=eget=0; armed=NULL;
               };
            }
            else
            {
               escstate=FALSE; epos=eget=0;
            };
         };
      };
   }
   else                          /* normal  output        */
   {
      if (out <= ASCII_CTL_MAX)  /* control character     */
      {
         for (i=0; i<(sizeof(ctlfnc)/sizeof(ctlfnc[0])); i++)
         {
            if (ctlfnc[i].key==out)
            {
               (*ctlfnc[i].ctlfunc)();
               break;
            };
         };
      }
      else
      {
#ifdef KOI7
         c=IsTtoKoi7() ? pdp2pcK7[out]:pdp2pcAS[out];
#else
         c=pdp2pcAS[out];
#endif
         if (c!=0) putch(c);
      };
   };
   return SCPE_OK;
};


/*
   VT52 Emulation support
*/

static void call_null(void)
{
};

static void ctl_bell(void)
{
   putch(0x07);
};

static void ctl_bs(void)
{
   if (wherex() > MIN_COL)
      gotoxy(wherex()-1,wherey());
};

static void ctl_ht(void)
{short tnum,posx;
   if (wherex() < MAX_COL-TAB_SIZE)
   {
      tnum=(wherex()-1)/TAB_SIZE; posx=(tnum+1)*TAB_SIZE;
   }
   else
      posx=MAX_COL;
   posx-=wherex();
   for (tnum=0; tnum<posx; tnum++) putch(' ');
};

static void ctl_lf(void)
{
   putch(0x0A);
};

static void ctl_vt(void)
{
   putch(0x0A);
};

static void ctl_ff(void)
{
   clrscr(); gotoxy(MIN_COL,MIN_ROW);
};

static void ctl_cr(void)
{
   putch(0x0D);
};

static void ctl_can(void)
{
   escstate=FALSE; epos=eget=0; armed=NULL;
};

static void ctl_sub(void)
{
   escstate=FALSE; epos=eget=0; armed=NULL;
};

static void ctl_esc(void)
{
   escstate=TRUE; epos=eget=0; armed=NULL;
};

static void esc_a(void)
{
   if (wherey() > MIN_ROW) gotoxy(wherex(),wherey()-1);
};

static void esc_b(void)
{
   if (wherey() < MAX_ROW) gotoxy(wherex(),wherey()+1);
};

static void esc_c(void)
{
   if (wherex() < MAX_COL) gotoxy(wherex()+1,wherey());
};

static void esc_d(void)
{
   if (wherex() > MIN_COL) gotoxy(wherex()-1,wherey());
};

static void esc_h(void)
{
   gotoxy(MIN_COL,MIN_ROW);
};

static void esc_i(void)
{union REGS in,out;
 short row,col;
   col=wherex(); row=wherey();
   if (row > MIN_ROW)
      gotoxy(col,row-1);
   else
   {
      in.h.ah=0x07; in.h.al=1; in.h.bh=attr;
      in.h.ch=1;               in.h.cl=0;
      in.h.dh=ScreenRows()-1;  in.h.dl=ScreenCols()-1;
      int86(0x10,&in,&out);
      gotoxy(col,row);
   };
};

static void esc_j(void)
{union REGS in,out;
 short col,row;
   col=wherex(); row=wherey();
   if (row<MAX_ROW)  /* clear rest of the screen */
   {
      in.h.ah=0x06; in.h.al=0; in.h.bh=attr;
      in.h.ch=row;             in.h.cl=0;
      in.h.dh=ScreenRows()-1;  in.h.dl=ScreenCols()-1;
      int86(0x10,&in,&out);
   };
   if (col<=MAX_COL) /* clear rest of the string */
   {
      in.h.ah=0x06; in.h.al=0; in.h.bh=attr;
      in.h.ch=row-1;           in.h.cl=col-1;
      in.h.dh=row-1;           in.h.dl=ScreenCols()-1;
      int86(0x10,&in,&out);
   };
   gotoxy(col,row);
};

static void esc_k(void)
{union REGS in,out;
 short col,row;
   col=wherex(); row=wherey();
   if (col<=MAX_COL) /* clear rest of the string */
   {
      in.h.ah=0x06; in.h.al=0; in.h.bh=attr;
      in.h.ch=row-1;           in.h.cl=col-1;
      in.h.dh=row-1;           in.h.dl=ScreenCols()-1;
      int86(0x10,&in,&out);
   };
   gotoxy(col,row);
};

static void esc_y(void)
{short row,col;
   row=escseq[1]-' '+1; col=escseq[2]-' '+1;
   gotoxy(col,row);
};

static void esc_z(void)
{
   inchar="\033/Z";
};

static void esc_eq(void) /* numlock off */
{

};

static void esc_gt(void) /* numlock on  */
{
};

static void esc_lbrk(void)
{
};

static void esc_slash(void)
{
};


/* EOF : SCP_TTY.C */

